﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/s3vectors/S3VectorsRequest.h>
#include <aws/s3vectors/S3Vectors_EXPORTS.h>

#include <utility>

namespace Aws {
namespace S3Vectors {
namespace Model {

/**
 */
class GetVectorsRequest : public S3VectorsRequest {
 public:
  AWS_S3VECTORS_API GetVectorsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "GetVectors"; }

  AWS_S3VECTORS_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The name of the vector bucket that contains the vector index. </p>
   */
  inline const Aws::String& GetVectorBucketName() const { return m_vectorBucketName; }
  inline bool VectorBucketNameHasBeenSet() const { return m_vectorBucketNameHasBeenSet; }
  template <typename VectorBucketNameT = Aws::String>
  void SetVectorBucketName(VectorBucketNameT&& value) {
    m_vectorBucketNameHasBeenSet = true;
    m_vectorBucketName = std::forward<VectorBucketNameT>(value);
  }
  template <typename VectorBucketNameT = Aws::String>
  GetVectorsRequest& WithVectorBucketName(VectorBucketNameT&& value) {
    SetVectorBucketName(std::forward<VectorBucketNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the vector index.</p>
   */
  inline const Aws::String& GetIndexName() const { return m_indexName; }
  inline bool IndexNameHasBeenSet() const { return m_indexNameHasBeenSet; }
  template <typename IndexNameT = Aws::String>
  void SetIndexName(IndexNameT&& value) {
    m_indexNameHasBeenSet = true;
    m_indexName = std::forward<IndexNameT>(value);
  }
  template <typename IndexNameT = Aws::String>
  GetVectorsRequest& WithIndexName(IndexNameT&& value) {
    SetIndexName(std::forward<IndexNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN of the vector index.</p>
   */
  inline const Aws::String& GetIndexArn() const { return m_indexArn; }
  inline bool IndexArnHasBeenSet() const { return m_indexArnHasBeenSet; }
  template <typename IndexArnT = Aws::String>
  void SetIndexArn(IndexArnT&& value) {
    m_indexArnHasBeenSet = true;
    m_indexArn = std::forward<IndexArnT>(value);
  }
  template <typename IndexArnT = Aws::String>
  GetVectorsRequest& WithIndexArn(IndexArnT&& value) {
    SetIndexArn(std::forward<IndexArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The names of the vectors you want to return attributes for. </p>
   */
  inline const Aws::Vector<Aws::String>& GetKeys() const { return m_keys; }
  inline bool KeysHasBeenSet() const { return m_keysHasBeenSet; }
  template <typename KeysT = Aws::Vector<Aws::String>>
  void SetKeys(KeysT&& value) {
    m_keysHasBeenSet = true;
    m_keys = std::forward<KeysT>(value);
  }
  template <typename KeysT = Aws::Vector<Aws::String>>
  GetVectorsRequest& WithKeys(KeysT&& value) {
    SetKeys(std::forward<KeysT>(value));
    return *this;
  }
  template <typename KeysT = Aws::String>
  GetVectorsRequest& AddKeys(KeysT&& value) {
    m_keysHasBeenSet = true;
    m_keys.emplace_back(std::forward<KeysT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Indicates whether to include the vector data in the response. The default
   * value is <code>false</code>.</p>
   */
  inline bool GetReturnData() const { return m_returnData; }
  inline bool ReturnDataHasBeenSet() const { return m_returnDataHasBeenSet; }
  inline void SetReturnData(bool value) {
    m_returnDataHasBeenSet = true;
    m_returnData = value;
  }
  inline GetVectorsRequest& WithReturnData(bool value) {
    SetReturnData(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Indicates whether to include metadata in the response. The default value is
   * <code>false</code>.</p>
   */
  inline bool GetReturnMetadata() const { return m_returnMetadata; }
  inline bool ReturnMetadataHasBeenSet() const { return m_returnMetadataHasBeenSet; }
  inline void SetReturnMetadata(bool value) {
    m_returnMetadataHasBeenSet = true;
    m_returnMetadata = value;
  }
  inline GetVectorsRequest& WithReturnMetadata(bool value) {
    SetReturnMetadata(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_vectorBucketName;

  Aws::String m_indexName;

  Aws::String m_indexArn;

  Aws::Vector<Aws::String> m_keys;

  bool m_returnData{false};

  bool m_returnMetadata{false};
  bool m_vectorBucketNameHasBeenSet = false;
  bool m_indexNameHasBeenSet = false;
  bool m_indexArnHasBeenSet = false;
  bool m_keysHasBeenSet = false;
  bool m_returnDataHasBeenSet = false;
  bool m_returnMetadataHasBeenSet = false;
};

}  // namespace Model
}  // namespace S3Vectors
}  // namespace Aws
