import requests
from typing import Dict, Optional

from django.urls import reverse
from django.utils.http import urlencode

from allauth.account.models import EmailAddress
from allauth.socialaccount.adapter import get_adapter
from allauth.socialaccount.internal import jwtkit
from allauth.socialaccount.providers.base import ProviderAccount
from allauth.socialaccount.providers.oauth2.client import OAuth2Error
from allauth.socialaccount.providers.oauth2.provider import OAuth2Provider
from allauth.socialaccount.providers.openid_connect.views import (
    OpenIDConnectOAuth2Adapter,
)


def _pick_data(data: dict) -> dict:
    # Prefer userinfo, it likely has more info compared to the ID token.
    if userinfo := data.get("userinfo"):
        return userinfo
    elif id_token := data.get("id_token"):
        return id_token
    # For backwards compatibility with <65.11
    return data


class OpenIDConnectProviderAccount(ProviderAccount):
    def get_user_data(self) -> Optional[Dict]:
        return _pick_data(self.account.extra_data)


class OpenIDConnectProvider(OAuth2Provider):
    id = "openid_connect"
    name = "OpenID Connect"
    account_class = OpenIDConnectProviderAccount
    oauth2_adapter_class = OpenIDConnectOAuth2Adapter
    supports_token_authentication = True

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.name = self.app.name

    @property
    def server_url(self):
        url = self.app.settings["server_url"]
        return self.wk_server_url(url)

    def wk_server_url(self, url):
        well_known_uri = "/.well-known/openid-configuration"
        if "/.well-known/" not in url:
            url += well_known_uri
        return url

    def get_login_url(self, request, **kwargs):
        url = reverse(
            f"{self.app.provider}_login", kwargs={"provider_id": self.app.provider_id}
        )
        if kwargs:
            url = f"{url}?{urlencode(kwargs)}"
        return url

    def get_callback_url(self):
        return reverse(
            f"{self.app.provider}_callback",
            kwargs={"provider_id": self.app.provider_id},
        )

    @property
    def token_auth_method(self):
        return self.app.settings.get("token_auth_method")

    def get_default_scope(self):
        return ["openid", "profile", "email"]

    def extract_uid(self, data):
        data = _pick_data(data)
        return str(data[self.app.settings.get("uid_field", "sub")])

    def extract_common_fields(self, data):
        data = _pick_data(data)
        return dict(
            email=data.get("email"),
            username=data.get("preferred_username"),
            name=data.get("name"),
            user_id=data.get("user_id"),
            picture=data.get("picture"),
            last_name=data.get("family_name"),
            first_name=data.get("given_name"),
        )

    def extract_email_addresses(self, data):
        data = _pick_data(data)
        addresses = []
        email = data.get("email")
        if email:
            addresses.append(
                EmailAddress(
                    email=email,
                    verified=data.get("email_verified", False),
                    primary=True,
                )
            )
        return addresses

    def get_oauth2_adapter(self, request):
        return self.oauth2_adapter_class(request, self.app.provider_id)

    def verify_token(self, request, token):
        id_token = token.get("id_token")
        if not id_token:
            raise get_adapter().validation_error("invalid_token")
        try:
            oauth2_adapter = self.get_oauth2_adapter(request)
            openid_config = oauth2_adapter.openid_config
            identity_data = jwtkit.verify_and_decode(
                credential=id_token,
                keys_url=openid_config["jwks_uri"],
                issuer=openid_config["issuer"],
                audience=[self.app.client_id],
                lookup_kid=jwtkit.lookup_kid_jwk,
            )
        except (OAuth2Error, requests.RequestException) as e:
            raise get_adapter().validation_error("invalid_token") from e
        login = self.sociallogin_from_response(request, identity_data)
        return login


provider_classes = [OpenIDConnectProvider]
