// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CONTENT_BROWSER_INDEXED_DB_FILE_PATH_UTIL_H_
#define CONTENT_BROWSER_INDEXED_DB_FILE_PATH_UTIL_H_

#include <stdint.h>

#include <string_view>

#include "base/files/file_path.h"
#include "base/functional/function_ref.h"
#include "content/common/content_export.h"

namespace storage {
struct BucketLocator;
}

namespace content::indexed_db {

inline constexpr base::FilePath::CharType kLevelDBExtension[] =
    FILE_PATH_LITERAL(".leveldb");
inline constexpr base::FilePath::CharType kIndexedDBExtension[] =
    FILE_PATH_LITERAL(".indexeddb");
inline constexpr base::FilePath::CharType kIndexedDBFile[] =
    FILE_PATH_LITERAL("indexeddb");

// Returns whether the legacy (first-party/default-bucket) path should be used
// for storing IDB files for the given bucket.
bool ShouldUseLegacyFilePath(const storage::BucketLocator& bucket_locator);

base::FilePath GetBlobStoreFileName(
    const storage::BucketLocator& bucket_locator);

base::FilePath GetLevelDBFileName(const storage::BucketLocator& bucket_locator);

base::FilePath GetBlobDirectoryName(const base::FilePath& path_base,
                                    int64_t database_id);

base::FilePath GetBlobDirectoryNameForKey(const base::FilePath& path_base,
                                          int64_t database_id,
                                          int64_t blob_number);

base::FilePath GetBlobFileNameForKey(const base::FilePath& path_base,
                                     int64_t database_id,
                                     int64_t blob_number);

// Returns if the given file path is too long for the current operating system's
// file system.
bool IsPathTooLong(const base::FilePath& path);

// Gets the relative path where all SQLite databases for the given bucket will
// be stored. This is relative to the `data_path` concept in BucketContext.
base::FilePath CONTENT_EXPORT
GetSqliteDbDirectory(const storage::BucketLocator& bucket_locator);

// The input correlates to any DOMString, which means it has arbitrary length
// and may include invalid UTF16. The output is a length-constrained, valid path
// component. This is only used for SQLite.
// TODO(crbug.com/419203257): add fuzz tests since the input is supplied by the
// page.
base::FilePath CONTENT_EXPORT
DatabaseNameToFileName(std::u16string_view db_name);

// Enumerates all files in `directory` that look like a database file, and
// applies `ref` to each. Specifically, if the name of the file is something
// that could have been generated by `DatabaseNameToFileName`, this will
// enumerate it. There is no guarantee that the enumerated file is actually a
// valid SQLite database.
void CONTENT_EXPORT EnumerateDatabasesInDirectory(
    const base::FilePath& directory,
    base::FunctionRef<void(const base::FilePath& path)> ref);

}  // namespace content::indexed_db

#endif  // CONTENT_BROWSER_INDEXED_DB_FILE_PATH_UTIL_H_
