// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CONTENT_BROWSER_INDEXED_DB_INSTANCE_SQLITE_RECORD_ITERATOR_H_
#define CONTENT_BROWSER_INDEXED_DB_INSTANCE_SQLITE_RECORD_ITERATOR_H_

#include <memory>

#include "content/browser/indexed_db/status.h"

namespace blink {
class IndexedDBKey;
}  // namespace blink

namespace sql {
class Statement;
}

namespace content::indexed_db {
class Record;

namespace sqlite {

// Iterates over a non-empty range of object store/index records, maintaining
// the position in the range and emitting `Record`s. `RecordIterator` is
// generated by the `DatabaseConnection` and owned by a
// `BackingStoreCursorImpl`. The actual SQLite resources (for statements) are
// owned by the `DatabaseConnection`, which also owns the `sql::Database`.
// Attempting to use the `RecordIterator` after the database connection is
// closed will result in error values being returned.
class RecordIterator {
 public:
  virtual ~RecordIterator();

  RecordIterator(const RecordIterator&) = delete;
  RecordIterator& operator=(const RecordIterator&) = delete;

  // Iterates from the current position until the target `key` and `primary_key`
  // are reached. Use when at least one of these is valid.
  StatusOr<std::unique_ptr<Record>> Iterate(
      const blink::IndexedDBKey& key,
      const blink::IndexedDBKey& primary_key);

  // Iterates `count` times from the current position.
  StatusOr<std::unique_ptr<Record>> Iterate(uint32_t count);

  // Saves the current position of the iterator.
  virtual void SavePosition() = 0;

  // Resets the iterator to the last saved position if there was one. Returns
  // false if no position was saved previously.
  virtual bool TryResetToLastSavedPosition() = 0;

 protected:
  RecordIterator();

  virtual void BindParameters(sql::Statement& statement,
                              const blink::IndexedDBKey& target_key,
                              const blink::IndexedDBKey& target_primary_key,
                              uint32_t offset) = 0;
  // Updates the current position in the range and returns the record from the
  // current row.
  virtual StatusOr<std::unique_ptr<Record>> ReadRow(
      sql::Statement& statement) = 0;
  // Returns the parsed and bound statement that embeds the SQL query for this
  // iterator. Because the records contained in the range can change between
  // `Iterate()` calls, the query needs to be typically re-executed every time.
  // The query itself is immutable for the duration of `this`, however, and
  // contains a mix of fixed and variable bound parameters. To avoid re-parsing
  // the query and rebinding the fixed parameters every time, the statement is
  // created once and registered with the `DatabaseConnection`.
  //
  // May return null if the SQLite database was closed e.g. due to an error.
  virtual sql::Statement* GetStatement() = 0;
};

}  // namespace sqlite
}  // namespace content::indexed_db

#endif  // CONTENT_BROWSER_INDEXED_DB_INSTANCE_SQLITE_RECORD_ITERATOR_H_
