// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "pdf/pdf_caret.h"

#include "base/check.h"
#include "base/check_op.h"
#include "base/containers/span.h"
#include "base/time/time.h"
#include "pdf/page_character_index.h"
#include "pdf/pdf_caret_client.h"
#include "pdf/region_data.h"
#include "third_party/skia/include/core/SkColor.h"
#include "ui/gfx/geometry/rect.h"

namespace chrome_pdf {

namespace {

constexpr SkColor4f kCaretColor = SkColors::kBlack;
constexpr int kCaretWidth = 1;

}  // namespace

PdfCaret::PdfCaret(PdfCaretClient* client, const PageCharacterIndex& index)
    : client_(client) {
  SetChar(index);
}

PdfCaret::~PdfCaret() = default;

void PdfCaret::SetVisibility(bool is_visible) {
  if (is_visible_ == is_visible) {
    return;
  }

  is_visible_ = is_visible;
  RefreshDisplayState();
}

void PdfCaret::SetBlinkInterval(base::TimeDelta interval) {
  if (interval.is_negative()) {
    return;
  }

  blink_interval_ = interval;
  RefreshDisplayState();
}

void PdfCaret::SetChar(const PageCharacterIndex& next_char) {
  uint32_t char_count = client_->GetCharCount(next_char.page_index);
  CHECK_GT(char_count, 0u);
  CHECK_LE(next_char.char_index, char_count);

  index_ = next_char;

  const gfx::Rect old_screen_rect = caret_screen_rect_;
  caret_screen_rect_ = GetScreenRectForCaret();
  if (!old_screen_rect.IsEmpty() && old_screen_rect != caret_screen_rect_) {
    client_->InvalidateRect(old_screen_rect);
  }
  if (is_visible_) {
    RefreshDisplayState();
  }
}

bool PdfCaret::MaybeDrawCaret(const RegionData& region,
                              const gfx::Rect& dirty_in_screen) const {
  if (!is_blink_visible_) {
    return false;
  }

  gfx::Rect visible_caret =
      gfx::IntersectRects(caret_screen_rect_, dirty_in_screen);
  if (visible_caret.IsEmpty()) {
    return false;
  }

  visible_caret.Offset(-dirty_in_screen.OffsetFromOrigin());
  Draw(region, visible_caret);
  return true;
}

void PdfCaret::OnGeometryChanged() {
  if (!is_visible_) {
    return;
  }

  caret_screen_rect_ = GetScreenRectForCaret();
  client_->InvalidateRect(caret_screen_rect_);
}

void PdfCaret::RefreshDisplayState() {
  blink_timer_.Stop();
  if (is_visible_ && blink_interval_.is_positive()) {
    blink_timer_.Start(FROM_HERE, blink_interval_, this,
                       &PdfCaret::OnBlinkTimerFired);
  }
  is_blink_visible_ = is_visible_;
  client_->InvalidateRect(caret_screen_rect_);
}

void PdfCaret::OnBlinkTimerFired() {
  CHECK(is_visible_);
  CHECK(blink_interval_.is_positive());
  is_blink_visible_ = !is_blink_visible_;
  client_->InvalidateRect(caret_screen_rect_);
}

gfx::Rect PdfCaret::GetScreenRectForCaret() const {
  gfx::Rect screen_rect;
  PageCharacterIndex curr_char = index_;

  do {
    screen_rect = GetScreenRectForChar(curr_char);
    if (!screen_rect.IsEmpty()) {
      break;
    }
    // Synthetic whitespaces and newlines generated by PDFium may not have a
    // screen rect. They cannot be the first char.
    CHECK_GT(curr_char.char_index, 0u);
    --curr_char.char_index;
  } while (true);

  if (index_.char_index != curr_char.char_index) {
    screen_rect.Offset(screen_rect.width(), 0);
  }

  CHECK(!screen_rect.IsEmpty());
  screen_rect.set_width(kCaretWidth);
  return screen_rect;
}

gfx::Rect PdfCaret::GetScreenRectForChar(
    const PageCharacterIndex& index) const {
  uint32_t char_count = client_->GetCharCount(index.page_index);
  CHECK_GT(char_count, 0u);
  CHECK_LE(index.char_index, char_count);
  if (index.char_index == char_count) {
    return gfx::Rect();
  }

  const std::vector<gfx::Rect> screen_rects =
      client_->GetScreenRectsForChar(index);
  return !screen_rects.empty() ? screen_rects[0] : gfx::Rect();
}

void PdfCaret::Draw(const RegionData& region, const gfx::Rect& rect) const {
  int l = rect.x();
  int t = rect.y();
  int w = rect.width();
  int h = rect.height();
  for (int y = t; y < t + h; ++y) {
    base::span<uint8_t> row =
        region.buffer.subspan(y * region.stride, region.stride);
    for (int x = l; x < l + w; ++x) {
      size_t pixel_index = x * 4;
      if (pixel_index + 2 < row.size()) {
        row[pixel_index] =
            static_cast<uint8_t>(row[pixel_index] * kCaretColor.fB);
        row[pixel_index + 1] =
            static_cast<uint8_t>(row[pixel_index + 1] * kCaretColor.fG);
        row[pixel_index + 2] =
            static_cast<uint8_t>(row[pixel_index + 2] * kCaretColor.fR);
      }
    }
  }
}

}  // namespace chrome_pdf
