-- Copyright (C) 2010-2011 John Millikin <jmillikin@gmail.com>
-- 
-- This program is free software: you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation, either version 3 of the License, or
-- any later version.
-- 
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
-- 
-- You should have received a copy of the GNU General Public License
-- along with this program.  If not, see <http://www.gnu.org/licenses/>.

module Network.Protocol.XMPP.Handle
	( Handle (..)
	, startTLS
	, hPutBytes
	, hGetBytes
	, handleIsSecure
	) where

import           Control.Monad (when, void)
import qualified Control.Monad.Trans.Except as E
import           Control.Monad.IO.Class (liftIO)
import qualified Data.ByteString
import           Data.ByteString (ByteString)
import qualified Data.ByteString.Lazy
import qualified Data.Text
import           Data.Text (Text)
import qualified System.IO as IO
import qualified Network.Protocol.TLS.GNU as TLS
import           Network.Protocol.XMPP.String (s)

data Handle =
	  PlainHandle IO.Handle
	| SecureHandle IO.Handle TLS.Session

liftTLS :: TLS.Session -> TLS.TLS a -> E.ExceptT Text IO a
liftTLS session = liftTLS' . TLS.runTLS session

liftTLS' :: IO (Either TLS.Error a) -> E.ExceptT Text IO a
liftTLS' io = do
	eitherX <- liftIO io
	case eitherX of
		Left err -> E.throwE (Data.Text.pack (show err))
		Right x -> return x

startTLS :: Handle -> E.ExceptT Text IO Handle
startTLS (SecureHandle _ _) = E.throwE $ s"Can't start TLS on a secure handle"
startTLS (PlainHandle h) = liftTLS' $ TLS.runClient (TLS.handleTransport h) $ do
	TLS.setCredentials =<< TLS.certificateCredentials
	TLS.handshake
	SecureHandle h `fmap` TLS.getSession

hPutBytes :: Handle -> ByteString -> E.ExceptT Text IO ()
hPutBytes (PlainHandle h)  = liftIO . Data.ByteString.hPut h
hPutBytes (SecureHandle _ session) = liftTLS session . TLS.putBytes . toLazy where
	toLazy bytes = Data.ByteString.Lazy.fromChunks [bytes]

hGetBytes :: Handle -> Integer -> E.ExceptT Text IO ByteString
hGetBytes (PlainHandle h) n = liftIO (Data.ByteString.hGet h (fromInteger n))
hGetBytes (SecureHandle h session) n = liftTLS session $ do
	pending <- TLS.checkPending
	let wait = void $ IO.hWaitForInput h (- 1)
	when (pending == 0) (liftIO wait)
	Data.ByteString.concat . Data.ByteString.Lazy.toChunks <$> getBytes
	where
	getBytes = TLS.getBytes n `E.catchE` handleGetBytesErr
	handleGetBytesErr (TLS.Error (-28)) = getBytes
	handleGetBytesErr e = E.throwE e

handleIsSecure :: Handle -> Bool
handleIsSecure PlainHandle{} = False
handleIsSecure SecureHandle{} = True
