﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/redshift/RedshiftRequest.h>
#include <aws/redshift/Redshift_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Redshift {
namespace Model {

/**
 */
class DescribeClusterTracksRequest : public RedshiftRequest {
 public:
  AWS_REDSHIFT_API DescribeClusterTracksRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DescribeClusterTracks"; }

  AWS_REDSHIFT_API Aws::String SerializePayload() const override;

 protected:
  AWS_REDSHIFT_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The name of the maintenance track. </p>
   */
  inline const Aws::String& GetMaintenanceTrackName() const { return m_maintenanceTrackName; }
  inline bool MaintenanceTrackNameHasBeenSet() const { return m_maintenanceTrackNameHasBeenSet; }
  template <typename MaintenanceTrackNameT = Aws::String>
  void SetMaintenanceTrackName(MaintenanceTrackNameT&& value) {
    m_maintenanceTrackNameHasBeenSet = true;
    m_maintenanceTrackName = std::forward<MaintenanceTrackNameT>(value);
  }
  template <typename MaintenanceTrackNameT = Aws::String>
  DescribeClusterTracksRequest& WithMaintenanceTrackName(MaintenanceTrackNameT&& value) {
    SetMaintenanceTrackName(std::forward<MaintenanceTrackNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An integer value for the maximum number of maintenance tracks to return.</p>
   */
  inline int GetMaxRecords() const { return m_maxRecords; }
  inline bool MaxRecordsHasBeenSet() const { return m_maxRecordsHasBeenSet; }
  inline void SetMaxRecords(int value) {
    m_maxRecordsHasBeenSet = true;
    m_maxRecords = value;
  }
  inline DescribeClusterTracksRequest& WithMaxRecords(int value) {
    SetMaxRecords(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An optional parameter that specifies the starting point to return a set of
   * response records. When the results of a <code>DescribeClusterTracks</code>
   * request exceed the value specified in <code>MaxRecords</code>, Amazon Redshift
   * returns a value in the <code>Marker</code> field of the response. You can
   * retrieve the next set of response records by providing the returned marker value
   * in the <code>Marker</code> parameter and retrying the request. </p>
   */
  inline const Aws::String& GetMarker() const { return m_marker; }
  inline bool MarkerHasBeenSet() const { return m_markerHasBeenSet; }
  template <typename MarkerT = Aws::String>
  void SetMarker(MarkerT&& value) {
    m_markerHasBeenSet = true;
    m_marker = std::forward<MarkerT>(value);
  }
  template <typename MarkerT = Aws::String>
  DescribeClusterTracksRequest& WithMarker(MarkerT&& value) {
    SetMarker(std::forward<MarkerT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_maintenanceTrackName;

  int m_maxRecords{0};

  Aws::String m_marker;
  bool m_maintenanceTrackNameHasBeenSet = false;
  bool m_maxRecordsHasBeenSet = false;
  bool m_markerHasBeenSet = false;
};

}  // namespace Model
}  // namespace Redshift
}  // namespace Aws
