use glib::{Properties, subclass};
use gtk4::{Align, CompositeTemplate, Label, Widget, prelude::*, subclass::prelude::*};
use libadwaita::{Bin, Easing, PropertyAnimationTarget, TimedAnimation, prelude::*, subclass::prelude::*};
use std::cell::{Cell, RefCell};

mod imp {
    use super::*;

    #[derive(Debug, Default, CompositeTemplate, Properties)]
    #[properties(wrapper_type = super::UrlOverlay)]
    #[template(file = "data/resources/ui_templates/article_view/url.blp")]
    pub struct UrlOverlay {
        #[template_child]
        pub overlay: TemplateChild<Bin>,
        #[template_child]
        pub label: TemplateChild<Label>,

        #[property(get, set, nullable)]
        pub child: RefCell<Option<Widget>>,

        #[property(get, set = Self::set_url, nullable)]
        pub url: RefCell<Option<String>>,

        pub(super) pointer_pos: Cell<(f64, f64)>,
        pub(super) fade_animation: RefCell<Option<TimedAnimation>>,
    }

    #[glib::object_subclass]
    impl ObjectSubclass for UrlOverlay {
        const NAME: &'static str = "UrlOverlay";
        type ParentType = Bin;
        type Type = super::UrlOverlay;

        fn class_init(klass: &mut Self::Class) {
            klass.bind_template();
            klass.bind_template_callbacks();
        }

        fn instance_init(obj: &subclass::InitializingObject<Self>) {
            obj.init_template();
        }
    }

    #[glib::derived_properties]
    impl ObjectImpl for UrlOverlay {
        fn constructed(&self) {
            let target = PropertyAnimationTarget::new(&*self.overlay, "opacity");
            let animation = TimedAnimation::builder()
                .target(&target)
                .widget(&*self.overlay)
                .duration(200)
                .value_from(0.0)
                .value_to(1.0)
                .easing(Easing::EaseInCubic)
                .build();

            animation.connect_done(|animation| {
                animation.widget().set_visible(animation.value() != 0.0);
            });

            self.fade_animation.replace(Some(animation));
        }
    }

    impl WidgetImpl for UrlOverlay {}

    impl BinImpl for UrlOverlay {}

    #[gtk4::template_callbacks]
    impl UrlOverlay {
        #[template_callback]
        fn is_visible(&self, url: Option<String>) -> bool {
            url.is_some()
        }

        #[template_callback]
        fn on_motion(&self, x: f64, y: f64) {
            self.pointer_pos.set((x, y));
            self.calc_overlay_position();
        }

        fn set_url(&self, url: Option<String>) {
            if let Some(label) = url.as_deref() {
                self.label.set_label(label);
                self.calc_overlay_position();
            }

            if let Some(animation) = self.fade_animation.borrow().as_ref() {
                animation.set_reverse(url.is_none());
                animation.play();
            }

            self.url.replace(url);
        }

        fn calc_overlay_position(&self) {
            let overlay_width = self.overlay.width();
            let overlay_height = self.overlay.height() + self.overlay.margin_top() + self.overlay.margin_bottom();

            if overlay_width == 0 || overlay_height == 0 {
                return;
            }

            let overlay_width = f64::from(overlay_width);
            let overlay_height = f64::from(overlay_height);

            let width = f64::from(self.obj().width());
            let height = f64::from(self.obj().height());

            let rel_pointer_x = self.pointer_pos.get().0 / width;
            let rel_pointer_y = self.pointer_pos.get().1 / height;

            let rel_label_width = overlay_width / width;
            let rel_label_height = overlay_height / height;

            let can_h_dodge = rel_pointer_x < (1.0 - rel_label_width) || rel_pointer_x > rel_label_width;
            let height_threshold = 1.0 - (rel_label_height * 1.2);

            let v_align = if !can_h_dodge && rel_pointer_y > height_threshold {
                Align::Start
            } else {
                Align::End
            };

            let h_align =
                if v_align == Align::End && rel_pointer_y > height_threshold && rel_pointer_x < (1.0 - rel_label_width)
                {
                    Align::End
                } else {
                    Align::Start
                };

            let change_align = self.overlay.valign() != v_align || self.overlay.halign() != h_align;

            self.overlay.set_valign(v_align);
            self.overlay.set_halign(h_align);

            if change_align {
                self.overlay.queue_resize();
            }
        }
    }
}

glib::wrapper! {
    pub struct UrlOverlay(ObjectSubclass<imp::UrlOverlay>)
        @extends Widget, Bin;
}

impl Default for UrlOverlay {
    fn default() -> Self {
        glib::Object::new::<Self>()
    }
}
