// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

#include "../shim.bwd_kernel_fuse.h"
#include <aotriton/_internal/triton_kernel.h>
#include <aotriton/_internal/kernel_cluster.h>
#include <aotriton/cpp_tune.h>
#include <string_view>
#ifndef NDEBUG
#include <iostream>
#endif

#define CURRENT_ENTRY_PUBLIC Autotune_bwd_kernel_fuse__A0__F402

#define ARRAY_SIZE(array)  (sizeof(array) / sizeof(array[0]))

namespace { // Anonymous namespace

using namespace std::literals::string_view_literals;

#if AOTRITON_BUILD_FOR_TUNING
// PSels and Copts in JSON String
static const char* kernel_psels[] = {
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw"
};
static const char* kernel_copts[] = {
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw"
};
#endif

struct PerfFields {
  int16_t BLOCK_M;
    int16_t BLOCK_N;
};

static PerfFields image_perf_list [] = {
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 }
};

// u8R generates char8_t which is poorly supported almost everywhere.
constexpr std::string_view PACKAGE_PATH { R"xyzw(amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp32@16_64_0_F_T_0___gfx950)xyzw" };
constexpr std::string_view FUNC_NAME { R"xyzw(＊fp32@16_64_0_F_T_0)xyzw" };
constexpr std::string_view ARCH_NAME { R"xyzw(gfx950)xyzw" };

// Checksum can be confirmed with `echo -n '<string>' | b2sum -l 64`
// For example:
//   $ echo -n 'amd-gfx110x/flash/attn_fwd/FONLY__^bf16@16,128,False,False,0,0___gfx1100__P__32_32_0_2_False__CO__wave3_warp2_stg1-Gpu-gfx1100' | b2sum -l 64
//   c4b51ee645d79580  -
static AOTRITON_NS::TritonKernelCompactMeta meta_list[] = {
    { 0x16bb3529u, 0xc58492f5u, 0, 155 }, // 16bb3529c58492f5 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp32@16_64_0_F_T_0___gfx950__P__16_16__CO__wave1_warp2_stg1--Arch_gfx950
    { 0xfec8c5c0u, 0x15cfa81au, 0, 6 }, // fec8c5c015cfa81a = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp32@16_64_0_F_T_0___gfx950__P__16_16__CO__wave2_warp2_stg1--Arch_gfx950
    { 0x26ed86c0u, 0xc3168228u, 0, 29 }, // 26ed86c0c3168228 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp32@16_64_0_F_T_0___gfx950__P__16_16__CO__wave3_warp4_stg1--Arch_gfx950
    { 0xe118f12fu, 0xa579cd44u, 0, 86 }, // e118f12fa579cd44 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp32@16_64_0_F_T_0___gfx950__P__16_16__CO__wave4_warp2_stg1--Arch_gfx950
    { 0x747a7a42u, 0x2fbff647u, 23, 155 }, // 747a7a422fbff647 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp32@16_64_0_F_T_0___gfx950__P__32_16__CO__wave1_warp2_stg1--Arch_gfx950
    { 0x32d20d24u, 0x3b0bff21u, 23, 138 }, // 32d20d243b0bff21 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp32@16_64_0_F_T_0___gfx950__P__32_16__CO__wave1_warp4_stg1--Arch_gfx950
    { 0x04593d43u, 0x0e550f97u, 23, 6 }, // 04593d430e550f97 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp32@16_64_0_F_T_0___gfx950__P__32_16__CO__wave2_warp2_stg1--Arch_gfx950
    { 0x69a1aca4u, 0xf2709893u, 23, 109 }, // 69a1aca4f2709893 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp32@16_64_0_F_T_0___gfx950__P__32_16__CO__wave3_warp2_stg1--Arch_gfx950
    { 0xa9207b14u, 0x50cee232u, 23, 29 }, // a9207b1450cee232 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊fp32@16_64_0_F_T_0___gfx950__P__32_16__CO__wave3_warp4_stg1--Arch_gfx950
};

static constexpr int kTotalNumKernels = ARRAY_SIZE(meta_list);

using AOTRITON_NS::v3::flash::autotune::bwd_kernel_fuse_packed_string;

static AOTRITON_NS::TritonKernelCluster<kTotalNumKernels>
kernel_cluster(meta_list,
               bwd_kernel_fuse_packed_string);

static int8_t lut[1][7][7] =
{
// GPU gfx950_mod0
{{6,2,6,4,4,6,4},
 {5,2,6,4,3,8,8},
 {5,4,6,1,3,1,1},
 {8,4,0,1,1,1,4},
 {8,4,4,7,1,7,7},
 {8,4,4,7,7,7,7},
 {8,4,4,7,7,7,7}}
// End of GPU gfx950_mod0
}
;

}; // End of anonymous namespace

namespace AOTRITON_NS::v3::flash::autotune {

// using AOTRITON_NS::v2::flash::BwdKernelFuseContext;

void CURRENT_ENTRY_PUBLIC(BwdKernelFuseContext& context, int mod_number) {
#if AOTRITON_BUILD_FOR_TUNING
    int preferred_index = context._has_preferred_kernel;
    context._total_number_of_kernels = kTotalNumKernels;
#ifndef NDEBUG
    std::cerr << "Autotune_bwd_kernel_fuse__A0__F402 "
              << "kTotalNumKernels = " << kTotalNumKernels << " "
              << "_has_preferred_kernel = " << preferred_index << " "
              << std::endl;
#endif
    if (preferred_index != -1) {
        if (preferred_index >= kTotalNumKernels)
            return ;
        context.kernel_on_device = kernel_cluster.get(preferred_index);
        context.pp_args_index = 0;
        context.package_path = PACKAGE_PATH;
        context.func_name = FUNC_NAME;
        context.arch_name = ARCH_NAME;
        context._preferred_kernel_psels = kernel_psels[preferred_index];
        context._preferred_kernel_copts = kernel_copts[preferred_index];
        const auto& perf = image_perf_list[preferred_index];
        context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
        return ;
    }
#endif
    auto kernel_index = bwd_kernel_fuse__lut_lambda__0(*context.params, mod_number, lut);
    if (kernel_index < 0) {
      return ;
    }
    context.kernel_on_device = kernel_cluster.get(kernel_index);
    context.pp_args_index = 0;
    context.package_path = PACKAGE_PATH;
    context.func_name = FUNC_NAME;
    context.arch_name = ARCH_NAME;
#ifndef NDEBUG
    std::cerr << __FILE__ << " kernel_index = " << int(kernel_index) << std::endl;
#endif
    const auto& perf = image_perf_list[kernel_index];
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
}

#undef CURRENT_ENTRY_PUBLIC
}

// Human-readable Signature 
// Q = "*fp32:16"
// sm_scale = "fp32"
// L = "*fp32:16"
// stride_bz = 0
// stride_dbz = 0
// num_head_q = "i32"
// cu_seqlens_q = "*i32:16"
// num_seqlens = "i32"
// dropout_p = 0
// philox_seed_ptr = 0
// philox_offset1 = 0
// philox_offset2 = 0
// Window_left = 0
// BLOCK_DMODEL = 64
// CAUSAL_TYPE = 0
// ENABLE_DROPOUT = False
// PADDED_HEAD = True
// BIAS_TYPE = 0

// vim: set fileencoding=utf-8

