use crate::app::App;
use crate::gobject_models::GArticle;
use crate::util::{DateUtil, GTK_RESOURCE_FILE_ERROR, constants};
use gdk4::pango;
use rust_embed::RustEmbed;
use std::str;

#[derive(RustEmbed)]
#[folder = "data/resources/article_view"]
pub(super) struct ArticleViewResources;

pub struct ArticleBuilder;

impl ArticleBuilder {
    pub fn from_garticle(article: &GArticle, prefer_scraped_content: bool, for_export: bool) -> String {
        let settings = App::default().settings();
        let template_data = ArticleViewResources::get("article.html").expect(GTK_RESOURCE_FILE_ERROR);
        let template_str = str::from_utf8(template_data.data.as_ref()).expect(GTK_RESOURCE_FILE_ERROR);
        let mut template_string = template_str.to_owned();

        // A list of fonts we should try to use in order of preference
        // We will pass all of these to CSS in order
        let mut font_options: Vec<String> = Vec::new();
        let mut font_families: Vec<String> = Vec::new();
        let mut font_size: Option<i32> = None;

        // Try to use the configured font if it exists
        if settings.article_view().use_custom_font()
            && let Some(font_setting) = settings.article_view().font()
        {
            font_options.push(font_setting);
        }

        // If there is no configured font, or it's broken, use the system default font
        let font_system = App::default().desktop_settings().document_font();
        font_options.push(font_system);

        // Backup if the system font is broken too
        font_options.push("sans".to_owned());

        for font in font_options {
            let desc = pango::FontDescription::from_string(&font);
            if let Some(family) = desc.family() {
                font_families.push(family.to_string());
            }
            if font_size.is_none() && desc.size() > 0 {
                font_size = Some(desc.size());
            }
        }

        // if font size configured use it, otherwise use 12 as default
        let font_size = font_size.unwrap_or(12);

        let font_size = font_size / pango::SCALE;
        let font_family = font_families.join(", ");

        // $HTML
        if prefer_scraped_content {
            if let Some(html) = article.scraped_content() {
                template_string = template_string.replacen("$HTML", &html, 1);
            } else if let Some(html) = article.content() {
                template_string = template_string.replacen("$HTML", &html, 1);
            } else {
                template_string = template_string.replacen("$HTML", constants::NO_CONTENT, 1);
            }
        } else if let Some(html) = article.content() {
            template_string = template_string.replacen("$HTML", &html, 1);
        } else {
            template_string = template_string.replacen("$HTML", constants::NO_CONTENT, 1);
        }

        if for_export {
            let header = Self::prepare_header(article, font_size);
            template_string = template_string.replacen("$HEADER", &header, 1);
        } else {
            template_string = template_string.replacen("$HEADER", "", 1);
        }

        // $THEME
        let theme = settings.general().theme().as_str().to_owned();
        template_string = template_string.replacen("$THEME", &theme, 1);

        // $FONTFAMILY
        template_string = template_string.replacen("$FONTFAMILY", &font_family, 1);

        // $FONTSIZE
        template_string = template_string.replacen("$FONTSIZE", &format!("{font_size}"), 1);

        if for_export {
            let content_width = settings.article_view().content_width();
            let line_height = settings.article_view().line_height();

            let css_data = ArticleViewResources::get("style.css").expect(GTK_RESOURCE_FILE_ERROR);
            let css_str = str::from_utf8(css_data.data.as_ref()).expect(GTK_RESOURCE_FILE_ERROR);
            let css_string = css_str.to_string();
            let css_string = css_string.replacen("$CONTENT_WIDTH", &format!("{content_width}px"), 1);
            let css_string = css_string.replacen("$LINE_HEIGHT", &format!("{line_height}em"), 1);

            let additional_styles =
                "<style>header.post\n{{\npadding: 1rem 2rem;\n}}\ndiv.nfcontent\n{{\npadding: 1rem 2rem;\n}}\n</style>";

            template_string = template_string.replacen(
                "$CSS",
                &format!("{additional_styles}\n<style>\n{css_string}\n</style>"),
                1,
            );
        } else {
            template_string = template_string.replacen("$CSS", "<style>body{padding:0px 4px;}</style>", 1);
        }

        let mut mathjax_string = None;

        if let Some(feed_settings) = settings.get_feed_settings(article.feed_id().as_ref()) {
            // mathjax
            let inline_math = feed_settings
                .inline_math
                .as_ref()
                .map(|inline| format!("inlineMath: [['{inline}', '{inline}']]"))
                .unwrap_or_default();
            let mathjax_data = ArticleViewResources::get("mathjax/tex-svg.js").expect(GTK_RESOURCE_FILE_ERROR);
            let mathjax_js_str =
                str::from_utf8(mathjax_data.data.as_ref()).expect("Failed to load MATHJAX from resources");
            mathjax_string = Some(format!(
                r#"
                <script>
                    MathJax = {{
                        tex: {{{inline_math}}},
                        svg: {{fontCache: 'global'}}
                    }};
                </script>
                <script id="MathJax-script" async>
                    {mathjax_js_str}
                </script>
            "#
            ));
        }

        template_string = template_string.replacen("$MATHJAX", mathjax_string.as_deref().unwrap_or_default(), 1);

        template_string
    }

    fn prepare_header(article: &GArticle, font_size: i32) -> String {
        let header_data = if article.title().is_some() {
            ArticleViewResources::get("title_header.inc").expect(GTK_RESOURCE_FILE_ERROR)
        } else {
            ArticleViewResources::get("no_title_header.inc").expect(GTK_RESOURCE_FILE_ERROR)
        };
        let header_str = str::from_utf8(header_data.data.as_ref()).expect(GTK_RESOURCE_FILE_ERROR);
        let mut header_string = header_str.to_string();

        let time = DateUtil::format_time(article.date().as_ref());
        let date = DateUtil::format_date(article.date().as_ref());
        let formated_date = format!("{date} at {time}");
        let author_date = if let Some(author) = article.author() {
            format!("posted by: {author}, {formated_date}")
        } else {
            formated_date
        };

        // $AUTHOR / $DATE
        header_string = header_string.replacen("$AUTHOR", &author_date, 1);

        // $SMALLSIZE x2
        let small_size = font_size - 2;
        header_string = header_string.replacen("$SMALLSIZE", &format!("{small_size}"), 2);

        // $TITLE (and URL)
        if let Some(mut title) = article.title() {
            if let Some(article_url) = article.url() {
                title = format!("<a href=\"{article_url}\" target=\"_blank\">{title}</a>")
            }
            header_string = header_string.replacen("$TITLE", &title, 1);
        }

        // $LARGESIZE
        let large_size = font_size * 2;
        header_string = header_string.replacen("$LARGESIZE", &format!("{large_size}"), 1);

        // $FEED
        header_string = header_string.replacen("$FEED", article.feed_title().as_str(), 1);

        header_string
    }
}
