use crate::util::constants;
use glib::{Properties, prelude::*, subclass::prelude::*};
use serde::{Deserialize, Serialize};
use std::cell::{Cell, RefCell};
use std::default::Default;

mod imp {
    use super::*;

    #[derive(Default, Debug, Serialize, Deserialize, Properties)]
    #[properties(wrapper_type = super::ArticleViewSettings)]
    pub struct ArticleViewSettings {
        #[property(get, set, name = "use-custom-font")]
        #[serde(default)]
        pub use_custom_font: Cell<bool>,

        #[property(get, set, nullable)]
        #[serde(default)]
        pub font: RefCell<Option<String>>,

        #[property(get, set, name = "content-width", default = constants::DEFAULT_ARTICLE_CONTENT_WIDTH)]
        #[serde(default)]
        pub content_width: Cell<u32>,

        #[property(get, set, name = "line-height", default = constants::DEFAULT_ARTICLE_LINE_HEIGHT)]
        #[serde(default)]
        pub line_height: Cell<f32>,
    }

    #[glib::object_subclass]
    impl ObjectSubclass for ArticleViewSettings {
        const NAME: &'static str = "ArticleViewSettings";
        type Type = super::ArticleViewSettings;
    }

    #[glib::derived_properties]
    impl ObjectImpl for ArticleViewSettings {}
}

glib::wrapper! {
    pub struct ArticleViewSettings(ObjectSubclass<imp::ArticleViewSettings>);
}

impl Serialize for ArticleViewSettings {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: serde::Serializer,
    {
        self.imp().serialize(serializer)
    }
}

impl<'de> Deserialize<'de> for ArticleViewSettings {
    fn deserialize<D>(deserializer: D) -> Result<Self, D::Error>
    where
        D: serde::Deserializer<'de>,
    {
        let inner = imp::ArticleViewSettings::deserialize(deserializer)?;
        Ok(inner.into())
    }
}

impl From<imp::ArticleViewSettings> for ArticleViewSettings {
    fn from(inner: imp::ArticleViewSettings) -> Self {
        let line_height = inner.line_height.get();
        let content_width = inner.content_width.get();

        let line_height = if line_height == 0.0 {
            constants::DEFAULT_ARTICLE_LINE_HEIGHT
        } else {
            line_height
        };
        let content_width = if content_width == 0 {
            constants::DEFAULT_ARTICLE_CONTENT_WIDTH
        } else {
            content_width
        };

        glib::Object::builder()
            .property("use-custom-font", inner.use_custom_font.get())
            .property("font", inner.font.borrow().clone())
            .property("content-width", content_width)
            .property("line-height", line_height)
            .build()
    }
}

impl Default for ArticleViewSettings {
    fn default() -> Self {
        imp::ArticleViewSettings::default().into()
    }
}
